<?php

if (!function_exists('image_url')) {
    /**
     * Generate URL for uploaded images in assets/uploads folder
     * 
     * @param string $folder The upload folder (biodata, activities, etc.)
     * @param string $filename The image filename
     * @param int $width Placeholder width if image not found
     * @param int $height Placeholder height if image not found
     * @return string
     */
    function image_url($folder, $filename, $width = 300, $height = 200)
    {
        if (empty($filename)) {
            return base_url("placeholder/{$width}/{$height}");
        }
        
        // Check if file exists in assets/uploads folder
        $filePath = FCPATH . 'assets/uploads/' . $folder . '/' . $filename;
        if (!file_exists($filePath)) {
            return base_url("placeholder/{$width}/{$height}");
        }
        
        return base_url("assets/uploads/{$folder}/{$filename}");
    }
}

if (!function_exists('assets_image_url')) {
    /**
     * Simple helper to generate assets image URL
     * 
     * @param string $folder The upload folder (biodata, activities, etc.)
     * @param string $filename The image filename
     * @return string
     */
    function assets_image_url($folder, $filename)
    {
        if (empty($filename)) {
            return base_url('assets/images/placeholder.png');
        }
        
        return base_url("assets/uploads/{$folder}/{$filename}");
    }
}

if (!function_exists('safe_image')) {
    /**
     * Generate safe image tag with fallback
     * 
     * @param string $folder The upload folder
     * @param string $filename The image filename
     * @param string $alt Alt text
     * @param string $class CSS classes
     * @param int $width Image width
     * @param int $height Image height
     * @return string
     */
    function safe_image($folder, $filename, $alt = '', $class = '', $width = 300, $height = 200)
    {
        $imageUrl = image_url($folder, $filename, $width, $height);
        $fullImageUrl = assets_image_url($folder, $filename);
        
        return sprintf(
            '<img src="%s" alt="%s" class="%s image-clickable" data-full-image="%s" style="cursor: pointer;" onclick="showImageModal(\'%s\', \'%s\')" />',
            $imageUrl,
            esc($alt),
            esc($class),
            $fullImageUrl,
            $fullImageUrl,
            esc($alt)
        );
    }
}

if (!function_exists('clickable_image')) {
    /**
     * Generate clickable image with modal view
     * 
     * @param string $folder The upload folder
     * @param string $filename The image filename
     * @param string $alt Alt text
     * @param string $class CSS classes
     * @param int $width Thumbnail width
     * @param int $height Thumbnail height
     * @return string
     */
    function clickable_image($folder, $filename, $alt = '', $class = 'img-fluid', $width = 300, $height = 200)
    {
        if (empty($filename)) {
            return sprintf(
                '<img src="%s" alt="%s" class="%s" />',
                base_url("placeholder/{$width}/{$height}"),
                esc($alt),
                esc($class)
            );
        }
        
        $thumbnailUrl = image_url($folder, $filename, $width, $height);
        $fullImageUrl = assets_image_url($folder, $filename);
        
        return sprintf(
            '<img src="%s" alt="%s" class="%s image-clickable" data-full-image="%s" style="cursor: pointer;" onclick="showImageModal(\'%s\', \'%s\')" />',
            $thumbnailUrl,
            esc($alt),
            esc($class),
            $fullImageUrl,
            $fullImageUrl,
            esc($alt)
        );
    }
}